import pandas as pd
import numpy as np
import plotly.graph_objects as go
import plotly.express as px
from scipy.stats import gaussian_kde
import os
from svglib.svglib import svg2rlg
from reportlab.graphics import renderPDF

# Plots.py: reads CSV outputs and generates comparison plots
# Save via SVG→PDF, then remove SVG

# Plot settings
width = 600
height = 400
margin = dict(l=20, r=20, t=20, b=20)
font = dict(family='Times New Roman', size=24, color='black')

def plot_vary_n(csv_file: str, output_basename: str):
    df = pd.read_csv(csv_file)
    n_vals = sorted(df['param'].unique())
    # Gradient from blue -> purple -> red
    grad = px.colors.sample_colorscale(['#0000FF', '#800080', '#FF0000'], len(n_vals))

    y_theo = df['y_theo'].values
    xs = np.linspace(-4, 4, 200)

    fig = go.Figure()

    # Infinite width (solid black)
    kde_theo = gaussian_kde(y_theo)
    fig.add_trace(go.Scatter(
        x=xs, y=kde_theo(xs), mode='lines', name='∞-width',
        line=dict(color='black', dash='solid'),
        legendrank=2
    ))

    # Empirical KDEs (dotted gradient)
    for idx, n_val in enumerate(n_vals):
        arr = df[df['param'] == n_val]['y_emp'].values
        kde_emp = gaussian_kde(arr)
        fig.add_trace(go.Scatter(
            x=xs, y=kde_emp(xs), mode='lines', name=f'n={n_val}',
            line=dict(color=grad[idx], dash='dot'),
            legendrank=1
        ))

    fig.update_layout(
        font=font,
        width=width, height=height,
        margin=margin,
        plot_bgcolor='white', paper_bgcolor='white',
        xaxis=dict(range=[-4, 4], showgrid=True, gridcolor='lightgray', zeroline=True, zerolinecolor='lightgray'),
        yaxis=dict(showgrid=True, gridcolor='lightgray', zeroline=True, zerolinecolor='lightgray'),
        legend=dict(x=0.75, y=0.98)
    )

    svg_file = f"{output_basename}.svg"
    pdf_file = f"{output_basename}.pdf"
    fig.write_image(svg_file)
    drawing = svg2rlg(svg_file)
    renderPDF.drawToFile(drawing, pdf_file)
    os.remove(svg_file)


def plot_vary_H(csv_file: str, output_basename: str):
    df = pd.read_csv(csv_file)
    H_vals = sorted(df['param'].unique())
    # grad = px.colors.sample_colorscale(['#0000FF', '#800080', '#FF0000'], len(H_vals))
    # grad = px.colors.sample_colorscale(px.colors.sequential.Oranges, len(H_vals))
    grad = ['mediumpurple', 'indigo']

    xs = np.linspace(-20, 20, 200)
    fig = go.Figure()
    for idx, H_val in enumerate(H_vals):
        sub = df[df['param'] == H_val]
        kde_theo = gaussian_kde(sub['y_theo'].values)
        fig.add_trace(go.Scatter(
            x=xs, y=kde_theo(xs), mode='lines', name=f'∞-width, H={H_val}',
            line=dict(color=grad[idx], dash='solid'),
            legendrank=2
        ))
        fig.add_trace(go.Histogram(
            x=sub['y_emp'].values,
            histnorm='probability density',
            name=f'n=256, H={H_val}',
            marker_color=grad[idx],
            opacity=0.5,
            legendrank=1
        ))

    fig.update_layout(
        font=font,
        width=width, height=height,
        margin=margin,
        plot_bgcolor='white', paper_bgcolor='white',
        xaxis=dict(range=[-20, 20], showgrid=True, gridcolor='lightgray', zeroline=True, zerolinecolor='lightgray'),
        yaxis=dict(showgrid=True, gridcolor='lightgray', zeroline=True, zerolinecolor='lightgray'),
        legend=dict(x=0.6, y=0.98)
    )

    svg_file = f"{output_basename}.svg"
    pdf_file = f"{output_basename}.pdf"
    fig.write_image(svg_file)
    drawing = svg2rlg(svg_file)
    renderPDF.drawToFile(drawing, pdf_file)
    os.remove(svg_file)

if __name__ == '__main__':
    plot_vary_n('data_vary_n.csv', 'vary_n')
    plot_vary_H('data_vary_H.csv', 'vary_H')
